/*
 * Decompiled with CFR 0.152.
 */
package org.iq80.leveldb.table;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.Objects;
import org.iq80.leveldb.CompressionType;
import org.iq80.leveldb.Options;
import org.iq80.leveldb.env.WritableFile;
import org.iq80.leveldb.table.BlockBuilder;
import org.iq80.leveldb.table.BlockEntry;
import org.iq80.leveldb.table.BlockHandle;
import org.iq80.leveldb.table.BlockTrailer;
import org.iq80.leveldb.table.BytewiseComparator;
import org.iq80.leveldb.table.FilterBlockBuilder;
import org.iq80.leveldb.table.FilterPolicy;
import org.iq80.leveldb.table.Footer;
import org.iq80.leveldb.table.UserComparator;
import org.iq80.leveldb.util.PureJavaCrc32C;
import org.iq80.leveldb.util.Slice;
import org.iq80.leveldb.util.Slices;
import org.iq80.leveldb.util.Snappy;
import org.iq80.leveldb.util.ZLib;

public class TableBuilder {
    public static final long TABLE_MAGIC_NUMBER = -2646017456237118633L;
    private static final Charset CHARSET = Charset.forName("UTF-8");
    private final int blockRestartInterval;
    private final int blockSize;
    private final CompressionType compressionType;
    private final WritableFile file;
    private final BlockBuilder dataBlockBuilder;
    private final BlockBuilder indexBlockBuilder;
    private final FilterBlockBuilder filterPolicyBuilder;
    private Slice lastKey;
    private final UserComparator userComparator;
    private long entryCount;
    private boolean closed;
    private boolean pendingIndexEntry;
    private BlockHandle pendingHandle;
    private Slice compressedOutput;
    private long position;

    public TableBuilder(Options options, WritableFile file, UserComparator userComparator) {
        Objects.requireNonNull(options, "options is null");
        Objects.requireNonNull(file, "file is null");
        this.file = file;
        this.userComparator = userComparator;
        this.blockRestartInterval = options.blockRestartInterval();
        this.blockSize = options.blockSize();
        this.compressionType = options.compressionType();
        this.dataBlockBuilder = new BlockBuilder((int)Math.min((double)this.blockSize * 1.1, (double)options.maxFileSize()), this.blockRestartInterval, userComparator);
        int expectedNumberOfBlocks = 1024;
        this.indexBlockBuilder = new BlockBuilder(20 * expectedNumberOfBlocks, 1, userComparator);
        this.lastKey = Slices.EMPTY_SLICE;
        if (options.filterPolicy() != null) {
            this.filterPolicyBuilder = new FilterBlockBuilder((FilterPolicy)options.filterPolicy());
            this.filterPolicyBuilder.startBlock(0L);
        } else {
            this.filterPolicyBuilder = null;
        }
    }

    public long getEntryCount() {
        return this.entryCount;
    }

    public long getFileSize() {
        return this.position;
    }

    public void add(BlockEntry blockEntry) throws IOException {
        Objects.requireNonNull(blockEntry, "blockEntry is null");
        this.add(blockEntry.getKey(), blockEntry.getValue());
    }

    public void add(Slice key, Slice value) throws IOException {
        Objects.requireNonNull(key, "key is null");
        Objects.requireNonNull(value, "value is null");
        Preconditions.checkState(!this.closed, "table is finished");
        if (this.entryCount > 0L) assert (this.userComparator.compare(key, this.lastKey) > 0) : "key must be greater than last key";
        if (this.pendingIndexEntry) {
            Preconditions.checkState(this.dataBlockBuilder.isEmpty(), "Internal error: Table has a pending index entry but data block builder is empty");
            Slice shortestSeparator = this.userComparator.findShortestSeparator(this.lastKey, key);
            Slice handleEncoding = BlockHandle.writeBlockHandle(this.pendingHandle);
            this.indexBlockBuilder.add(shortestSeparator, handleEncoding);
            this.pendingIndexEntry = false;
        }
        if (this.filterPolicyBuilder != null) {
            this.filterPolicyBuilder.addKey(key);
        }
        this.lastKey = key;
        ++this.entryCount;
        this.dataBlockBuilder.add(key, value);
        int estimatedBlockSize = this.dataBlockBuilder.currentSizeEstimate();
        if (estimatedBlockSize >= this.blockSize) {
            this.flush();
        }
    }

    private void flush() throws IOException {
        Preconditions.checkState(!this.closed, "table is finished");
        if (this.dataBlockBuilder.isEmpty()) {
            return;
        }
        Preconditions.checkState(!this.pendingIndexEntry, "Internal error: Table already has a pending index entry to flush");
        this.pendingHandle = this.writeBlock(this.dataBlockBuilder);
        if (this.filterPolicyBuilder != null) {
            this.filterPolicyBuilder.startBlock(this.position);
        }
        this.pendingIndexEntry = true;
    }

    private BlockHandle writeBlock(BlockBuilder blockBuilder) throws IOException {
        Slice raw = blockBuilder.finish();
        BlockHandle blockHandle = this.writeRawBlock(raw);
        blockBuilder.reset();
        return blockHandle;
    }

    private BlockHandle writeRawBlock(Slice raw) throws IOException {
        Slice blockContents = raw;
        CompressionType blockCompressionType = CompressionType.NONE;
        if (this.compressionType == CompressionType.ZLIB || this.compressionType == CompressionType.ZLIB_RAW) {
            this.ensureCompressedOutputCapacity(TableBuilder.maxCompressedLength(raw.length()));
            try {
                int compressedSize = ZLib.compress(raw.getRawArray(), raw.getRawOffset(), raw.length(), this.compressedOutput.getRawArray(), 0, this.compressionType == CompressionType.ZLIB_RAW);
                if (compressedSize < raw.length() - raw.length() / 8) {
                    blockContents = this.compressedOutput.slice(0, compressedSize);
                    blockCompressionType = this.compressionType;
                }
            }
            catch (IOException compressedSize) {}
        } else if (this.compressionType == CompressionType.SNAPPY) {
            this.ensureCompressedOutputCapacity(TableBuilder.maxCompressedLength(raw.length()));
            try {
                int compressedSize = Snappy.compress(raw.getRawArray(), raw.getRawOffset(), raw.length(), this.compressedOutput.getRawArray(), 0);
                if (compressedSize < raw.length() - raw.length() / 8) {
                    blockContents = this.compressedOutput.slice(0, compressedSize);
                    blockCompressionType = CompressionType.SNAPPY;
                }
            }
            catch (IOException compressedSize) {
                // empty catch block
            }
        }
        BlockTrailer blockTrailer = new BlockTrailer(blockCompressionType, TableBuilder.crc32c(blockContents, blockCompressionType));
        Slice trailer = BlockTrailer.writeBlockTrailer(blockTrailer);
        BlockHandle blockHandle = new BlockHandle(this.position, blockContents.length());
        this.file.append(blockContents);
        this.file.append(trailer);
        this.position += (long)(blockContents.length() + trailer.length());
        return blockHandle;
    }

    private static int maxCompressedLength(int length) {
        return 32 + length + length / 6;
    }

    public void finish() throws IOException {
        Preconditions.checkState(!this.closed, "table is finished");
        this.flush();
        this.closed = true;
        BlockHandle filterBlockHandle = null;
        if (this.filterPolicyBuilder != null) {
            filterBlockHandle = this.writeRawBlock(this.filterPolicyBuilder.finish());
        }
        BlockBuilder metaIndexBlockBuilder = new BlockBuilder(256, this.blockRestartInterval, new BytewiseComparator());
        if (filterBlockHandle != null) {
            metaIndexBlockBuilder.add(new Slice(("filter." + this.filterPolicyBuilder.name()).getBytes(CHARSET)), BlockHandle.writeBlockHandle(filterBlockHandle));
        }
        BlockHandle metaindexBlockHandle = this.writeBlock(metaIndexBlockBuilder);
        if (this.pendingIndexEntry) {
            Slice shortSuccessor = this.userComparator.findShortSuccessor(this.lastKey);
            Slice handleEncoding = BlockHandle.writeBlockHandle(this.pendingHandle);
            this.indexBlockBuilder.add(shortSuccessor, handleEncoding);
            this.pendingIndexEntry = false;
        }
        BlockHandle indexBlockHandle = this.writeBlock(this.indexBlockBuilder);
        Footer footer = new Footer(metaindexBlockHandle, indexBlockHandle);
        Slice footerEncoding = Footer.writeFooter(footer);
        this.file.append(footerEncoding);
        this.position += (long)footerEncoding.length();
    }

    public void abandon() {
        this.closed = true;
    }

    public static int crc32c(Slice data, CompressionType type) {
        PureJavaCrc32C crc32c = new PureJavaCrc32C();
        crc32c.update(data.getRawArray(), data.getRawOffset(), data.length());
        crc32c.update(type.persistentId() & 0xFF);
        return crc32c.getMaskedValue();
    }

    public void ensureCompressedOutputCapacity(int capacity) {
        if (this.compressedOutput != null && this.compressedOutput.length() > capacity) {
            return;
        }
        this.compressedOutput = Slices.allocate(capacity);
    }
}

