/*
 * Decompiled with CFR 0.152.
 */
package org.iq80.leveldb.table;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import java.io.Closeable;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Comparator;
import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.iq80.leveldb.CompressionType;
import org.iq80.leveldb.DBException;
import org.iq80.leveldb.ReadOptions;
import org.iq80.leveldb.env.RandomInputFile;
import org.iq80.leveldb.iterator.SeekingIterators;
import org.iq80.leveldb.iterator.SliceIterator;
import org.iq80.leveldb.table.Block;
import org.iq80.leveldb.table.BlockHandle;
import org.iq80.leveldb.table.BlockIterator;
import org.iq80.leveldb.table.BlockTrailer;
import org.iq80.leveldb.table.BytewiseComparator;
import org.iq80.leveldb.table.CacheKey;
import org.iq80.leveldb.table.FilterBlockReader;
import org.iq80.leveldb.table.FilterPolicy;
import org.iq80.leveldb.table.Footer;
import org.iq80.leveldb.table.KeyValueFunction;
import org.iq80.leveldb.util.ILRUCache;
import org.iq80.leveldb.util.PureJavaCrc32C;
import org.iq80.leveldb.util.Slice;
import org.iq80.leveldb.util.Slices;
import org.iq80.leveldb.util.Snappy;
import org.iq80.leveldb.util.ZLib;

public final class Table
implements Closeable {
    private static final Charset CHARSET = StandardCharsets.UTF_8;
    private static final AtomicLong ID_GENERATOR = new AtomicLong();
    private final long id = ID_GENERATOR.incrementAndGet();
    private final Comparator<Slice> comparator;
    private final Block indexBlock;
    private final BlockHandle metaindexBlockHandle;
    private final RandomInputFile source;
    private final ILRUCache<CacheKey, Slice> blockCache;
    private final FilterBlockReader filter;
    private final AtomicInteger refCount = new AtomicInteger(1);

    public Table(RandomInputFile source, Comparator<Slice> comparator, boolean paranoidChecks, ILRUCache<CacheKey, Slice> blockCache, FilterPolicy filterPolicy) throws IOException {
        this.source = source;
        this.blockCache = blockCache;
        Objects.requireNonNull(source, "source is null");
        long size = source.size();
        Preconditions.checkArgument(size >= 48L, "File is corrupt: size must be at least %s bytes", 48);
        Objects.requireNonNull(comparator, "comparator is null");
        this.comparator = comparator;
        ByteBuffer footerData = source.read(size - 48L, 48);
        Footer footer = Footer.readFooter(Slices.avoidCopiedBuffer(footerData));
        this.indexBlock = new Block(this.readRawBlock(footer.getIndexBlockHandle(), paranoidChecks), comparator);
        this.metaindexBlockHandle = footer.getMetaindexBlockHandle();
        this.filter = this.readMeta(filterPolicy, paranoidChecks);
    }

    private FilterBlockReader readMeta(FilterPolicy filterPolicy, boolean verifyChecksum) throws IOException {
        assert (this.refCount.get() > 0);
        if (filterPolicy == null) {
            return null;
        }
        Block meta = new Block(this.readRawBlock(this.metaindexBlockHandle, verifyChecksum), new BytewiseComparator());
        try (BlockIterator iterator = meta.iterator();){
            Slice targetKey = new Slice(("filter." + filterPolicy.name()).getBytes(CHARSET));
            if (iterator.seek(targetKey) && ((Slice)iterator.key()).equals(targetKey)) {
                FilterBlockReader filterBlockReader = this.readFilter(filterPolicy, (Slice)iterator.value(), verifyChecksum);
                return filterBlockReader;
            }
            FilterBlockReader filterBlockReader = null;
            return filterBlockReader;
        }
    }

    protected FilterBlockReader readFilter(FilterPolicy filterPolicy, Slice filterHandle, boolean verifyChecksum) throws IOException {
        assert (this.refCount.get() > 0);
        Slice filterBlock = this.readRawBlock(BlockHandle.readBlockHandle(filterHandle.input()), verifyChecksum);
        return new FilterBlockReader(filterPolicy, filterBlock);
    }

    public SliceIterator iterator(ReadOptions options) {
        assert (this.refCount.get() > 0);
        this.retain();
        return SeekingIterators.twoLevelSliceIterator(this.indexBlock.iterator(), blockHandle -> this.openBlock(options, (Slice)blockHandle), this::release);
    }

    private BlockIterator openBlock(ReadOptions options, Slice blockHandle) {
        Block dataBlock = this.openBlock(blockHandle, options);
        return dataBlock.iterator();
    }

    public FilterBlockReader getFilter() {
        assert (this.refCount.get() > 0);
        return this.filter;
    }

    public Block openBlock(Slice blockEntry, ReadOptions options) {
        Block dataBlock;
        assert (this.refCount.get() > 0);
        BlockHandle blockHandle = BlockHandle.readBlockHandle(blockEntry.input());
        try {
            dataBlock = this.readBlock(blockHandle, options);
        }
        catch (IOException e) {
            throw new DBException(e);
        }
        return dataBlock;
    }

    private Block readBlock(BlockHandle blockHandle, ReadOptions options) throws IOException {
        assert (this.refCount.get() > 0);
        try {
            Slice ifPresent;
            Slice rawBlock = this.blockCache == null ? this.readRawBlock(blockHandle, options.verifyChecksums()) : (!options.fillCache() ? ((ifPresent = this.blockCache.getIfPresent(new CacheKey(this.id, blockHandle))) == null ? this.readRawBlock(blockHandle, options.verifyChecksums()) : ifPresent) : this.blockCache.load(new CacheKey(this.id, blockHandle), () -> this.readRawBlock(blockHandle, options.verifyChecksums())));
            return new Block(rawBlock, this.comparator);
        }
        catch (ExecutionException e) {
            Throwables.propagateIfPossible(e.getCause(), IOException.class);
            throw new IOException(e.getCause());
        }
    }

    protected Slice readRawBlock(BlockHandle blockHandle, boolean verifyChecksum) throws IOException {
        Slice uncompressedData;
        assert (this.refCount.get() > 0);
        ByteBuffer content = this.source.read(blockHandle.getOffset(), blockHandle.getFullBlockSize());
        int limit = content.limit();
        int position = content.position();
        int trailerStart = position + blockHandle.getDataSize();
        content.position(trailerStart);
        BlockTrailer blockTrailer = BlockTrailer.readBlockTrailer(Slices.avoidCopiedBuffer(content));
        if (verifyChecksum) {
            PureJavaCrc32C checksum = new PureJavaCrc32C();
            content.position(position).limit(trailerStart + 1);
            checksum.update(content);
            int actualCrc32c = checksum.getMaskedValue();
            Preconditions.checkState(blockTrailer.getCrc32c() == actualCrc32c, "Block corrupted: checksum mismatch");
        }
        content.position(position);
        content.limit(limit - 5);
        if (blockTrailer.getCompressionType() == CompressionType.ZLIB || blockTrailer.getCompressionType() == CompressionType.ZLIB_RAW) {
            ByteBuffer uncompressed = ZLib.uncompress(content, blockTrailer.getCompressionType() == CompressionType.ZLIB_RAW);
            uncompressedData = Slices.avoidCopiedBuffer(uncompressed);
        } else if (blockTrailer.getCompressionType() == CompressionType.SNAPPY) {
            ByteBuffer uncompressed = Snappy.uncompress(content);
            uncompressedData = Slices.copiedBuffer(uncompressed);
        } else {
            uncompressedData = Slices.avoidCopiedBuffer(content);
        }
        return uncompressedData;
    }

    public <T> T internalGet(ReadOptions options, Slice key, KeyValueFunction<T> keyValueFunction) {
        assert (this.refCount.get() > 0);
        try (BlockIterator iterator = this.indexBlock.iterator();){
            block17: {
                if (iterator.seek(key)) {
                    Slice handleValue = (Slice)iterator.value();
                    if (this.filter != null && !this.filter.keyMayMatch(BlockHandle.readBlockHandle(handleValue.input()).getOffset(), key)) {
                        T t = null;
                        return t;
                    }
                    BlockIterator iterator1 = this.openBlock(handleValue, options).iterator();
                    if (!iterator1.seek(key)) break block17;
                    T t = keyValueFunction.apply((Slice)iterator1.key(), (Slice)iterator1.value());
                    return t;
                    finally {
                        if (iterator1 != null) {
                            iterator1.close();
                        }
                    }
                }
            }
            T t = null;
            return t;
        }
    }

    public long getApproximateOffsetOf(Slice key) {
        assert (this.refCount.get() > 0);
        try (BlockIterator iterator = this.indexBlock.iterator();){
            if (iterator.seek(key)) {
                BlockHandle blockHandle = BlockHandle.readBlockHandle(((Slice)iterator.value()).input());
                long l = blockHandle.getOffset();
                return l;
            }
        }
        return this.metaindexBlockHandle.getOffset();
    }

    public boolean retain() {
        int refs;
        do {
            if ((refs = this.refCount.get()) != 0) continue;
            return false;
        } while (!this.refCount.compareAndSet(refs, refs + 1));
        return true;
    }

    public void release() throws IOException {
        assert (this.refCount.get() > 0);
        int refs = this.refCount.decrementAndGet();
        if (refs == 0) {
            this.source.close();
        }
    }

    public String toString() {
        return "Table{source='" + String.valueOf(this.source) + "', comparator=" + String.valueOf(this.comparator) + "}";
    }

    @Override
    public void close() throws IOException {
        this.release();
    }
}

